--isso é o lexer
--ele é responsavel por analisar o código/texto passando caracter por caracter
--identifica padrões como número/palavra/texto/símbolos
--o lexer é quem cria os tokens
--todo padrão que ele encontrado é um novo token

--exemplo
--esse é o código "var num = 0"
--o lexer passará por cada caracter usando o "cursor" como posição
--se o "cursor" for 1 então o caracter é "v"
--se for 2 então "a"
--se 3 então "r"
--e assim por diante
--vai passando caracter por caracter
--e fará algumas checagens para decidir oque fazer

--o lexer não entende o significado, ele apenas classifica partes do texto (número, palavra, símbolo, etc)
--e cria o token com o "kind" = tipo/nome/identificação e o "value" = fatia/código/texto/dado bruto

local Token = require("token")
local Keywords = require("keywords")

---@class Lexer
---@field __type "Lexer"
---@field source string --o código/texto que será processado
---@field cursor integer --a posição do caracter em "source"
local Lexer = {}
Lexer.__index = Lexer
Lexer.__type = "Lexer"
---cria um novo lexer
---@return Lexer
function Lexer.new()
    return setmetatable({
        source = "",
        cursor = 1
    }, Lexer)
end
---isso checa se é possivel mover/consumir o caracter atual
---@return boolean
function Lexer:canConsume()
    return self.cursor <= #self.source
end
---isso consome o caracter atual, move para a próxima, atribuindo 1 a "cursor"
function Lexer:consume()
    self.cursor = self.cursor + 1
end
---isso obtem o caracter atual usando o "cursor"
---@return string
function Lexer:peek()
    return self.source:sub(self.cursor, self.cursor)
end
---isso obtem uma fatia/pedaço iniciando de "start" até "cursor"
---@param start integer
---@return string
function Lexer:peekSlice(start)
    return self.source:sub(start, self.cursor - 1)
end
---isso é para checar um caracter com o atual
---@param char string
---@return boolean
function Lexer:check(char)
    if not self:canConsume() then
        return false
    end

    return self:peek() == char
end
---isso checa se o caracter atual é um digito
---isso é uma das checagem muito importante, sem isso não se pode identificar números
---@return boolean
function Lexer:isDigit()
    local char = self:peek()
    return char >= '0' and char <= '9'
end
---isso checa se o caracter atual é uma letra, podendo ser a-z ou A-Z
-- aqui só consideramos letras a-z e A-Z por simplicidade
---isso é muito importante, sem isso não é possivel identificar palavras
---@return boolean
function Lexer:isLetter()
    local char = self:peek()
    return (char >= 'a' and char <= 'z') or (char >= 'A' and char <= 'Z')
end
---isso checa se o caracter atual é o símbolo ' ou "
---isso ajuda a identificar o início de textos/strings
---exemplo "vermelho" ou 'pássaro' isso não é uma palavra conhecida pela linguagem, é um simples texto ou como chamamos 'string'
---isso também é bem importante
---@return boolean
function Lexer:isStringDelimiter()
   local char = self:peek()
   return char == '\'' or char == '"'
end
---isso checa se o caracter atual é um espaço ou '\n'(quebra de linha, nova linha) ou também '\t'(tabulação, tecla tab)
---isso é bom para checa o proprio espaço como considerar outros caracteres como espaços
---@return boolean
function Lexer:isWhitespace()
    local char = self:peek()
    return char == ' ' or char == '\n' or char == '\t'
end
---isso captura a sequencia de digitos como um número
---exemplo de código "123" isso será considerado como o número 123 (ainda como texto, depois será feita conversão)
---também é feita a checagem se possui "ponto", caso tenha mais de 1 "ponto" vai dar erro
---exemplo de código "123" isso é ok, "123.0" isso também é ok, mas "123..0" ou "1.2.3" var dar erro
---o resultado disso será usado para a criação do token {Number, VALOR CAPTURADO}
---@return string
function Lexer:consumeNumber()
    --guarda a posição inicial do número usando o "cursor"
    local start, dot = self.cursor, false
    --aqui ficará em loop enquanto poder consumir e o caracter for digito ou '.'
    while self:canConsume() and (self:isDigit() or self:peek() == '.') do
        if self:peek() == '.' then
            if dot then --essa checagem é para saber se o número tem mais de 1 ponto
                error("número malformado, encontrado dois pontos")
            end
            dot = true
        end

        self:consume()
    end

    return self:peekSlice(start)
end
---isso captura a sequencia de letras como uma palavra
---exemplo "vermelho" isso será considerado como a palavra vermelho, pois consome a palavra não
---o resultado disso será usado para a criação do token {Identifier, VALOR CAPTURADO} ou palavras reservadas como "var", "func", "true" entre outras
---@return string
function Lexer:consumeWord()
    --guarda a posição inicial da palavra usando o "cursor"
    local start = self.cursor
    --aqui ficará em loop enquanto poder consumir caracter e o caracter for letra(a-z ou A-Z) ou '_'
    while self:canConsume() and (self:isLetter() or self:peek() == '_') do
        self:consume()
    end

    return self:peekSlice(start)
end
---isso captura a sequencia de caracteres entre '' ou "" um(a) texto/string
---exemplo "vermelho" ou 'pássaro'
---o resultado disso será usado para a criação do token {String, VALOR CAPTURADO}
---@return string
function Lexer:consumeString()
    --guarda o delimitador, pode ser ' ou "
    local delimiter = self:peek()
    --consome o delimitador
    self:consume()
    --guarda a posição inicial do texto/string, e um booleano para saber se o texto foi fechado usando o delimitador
    local start, closed = self.cursor, false
    --aqui ficará em loop enquanto poder consumir caracter ou for fechado usando o delimitador
    while self:canConsume() do
        if self:peek() == delimiter then
            closed = true
            break
        end
        self:consume()
    end

    if closed then
        --guarda o texto/string de "start" até o "cursor"
        local str = self:peekSlice(start)
        --consome o delimitador
        self:consume()
        return str
    end
    --caso não sejá fechado usando o delimitador que foi usando no início dará erro
    error("não foi encontrado '" .. delimiter .. "' para fechar o texto")
end
---consome tudo que for considerado espaços
---são considerados espaços ' '(o proprio espaço kkk) também o '\n'(quebra de linha, nova linha) e o '\t'(tabulação, techa tab)
function Lexer:consumeWhitespace()
    --aqui ficará em loop enquanto poder consumir caracter e o caracter for considerado um espaço
    while self:canConsume() and self:isWhitespace() do
        self:consume()
    end
end
---aqui será processado o código/texto e terá a criação dos tokens
---@param source string
---@return Token[]
function Lexer:tokenize(source)
    self.source = source
    local tokens = {}

    --aqui ficará em loop enquanto poder consumir
    while self:canConsume() do
        --consome tudo que for considerado espaço
        self:consumeWhitespace()

        if not self:canConsume() then break end --caso após os espaços não tiver nada

        --agora tem algumas checagens, algumas se repetem muito

        if self:isDigit() then --se o caracter atual(usando o "cursor" como posição) for um digito
            --captura o número, cria o token e adiciona a lista de tokens
            table.insert(tokens, Token.new("Number", self:consumeNumber()))
        elseif self:isLetter() then --se o caracter atual for uma letra
            --captura a palavra
            local word = self:consumeWord()
            --checa se é uma palavra reservada, exemplo var/if/else/..
            local kind = Keywords[word]
            if kind then
                --cria o token e usa o "kind"(logo acima) e adiciona a lista de tokens
                table.insert(tokens, Token.new(kind, kind == "Boolean" and word or nil)) --caso o "kind" não sejá "Boolean", não é necessario usar o "value" do token
            else
                --cria o token sendo do tipo "Identifier" e adiciona a lista de tokens
                table.insert(tokens, Token.new("Identifier", word))
            end
        elseif self:isStringDelimiter() then --se o caracter atual for ' ou "
            --captura o texto/string, cria o token e adiciona a lista de tokens
            table.insert(tokens, Token.new("String", self:consumeString()))
        elseif self:check('+') then --se o caracter atual é '+'(símbolo mais)
            --adiciona na lista de tokens, também não é necessario usar o "value" do token
            table.insert(tokens, Token.new("Sum"))
            --consome o '+'
            self:consume()
        elseif self:check('-') then --aqui repete igual o '+' só muda o "kind" do token
            table.insert(tokens, Token.new("Minus"))
            self:consume()
        elseif self:check('*') then --aqui repete igual o '+' só muda o "kind" do token
            table.insert(tokens, Token.new("Asterisk"))
            self:consume()
        elseif self:check('/') then --aqui repete igual o '+' só muda o "kind" do token
            table.insert(tokens, Token.new("Slash"))
            self:consume()
        elseif self:check('=') then --aqui muda, são duas checagens, checa se é '='
            --consome o '='
            self:consume()
            --checa se tem um '=' novamente
            if self:check('=') then --caso tenha, então é um '==', caso não tenha então é '='
                --adiciona na lista de tokens o "Equal"(==)
                table.insert(tokens, Token.new("Equal"))
                --consome novamente o '='
                self:consume()
            else
                --adiciona na lista de tokens "Assign"(=)
                table.insert(tokens, Token.new("Assign"))
            end
        elseif self:check('>') then --semelhante o anterior, mas é com o ">"
            --consome o '>'
            self:consume()
            --checa se tem um '='
            if self:check('=') then --caso tenha, então é um '>=', caso não tenha então é '>'
                --adiciona na lista de tokens
                table.insert(tokens, Token.new("GreaterEqual"))
                --consome '='
                self:consume()
            else
                --adiciona na lista de tokens
                table.insert(tokens, Token.new("Greater"))
            end
        elseif self:check('<') then --semelhante o anterior, mas é com o "<"
            --consome o '<'
            self:consume()
            --checa se tem um '='
            if self:check('=') then --caso tenha, então é um '<=', caso não tenha então é '<'
                --adiciona na lista de tokens
                table.insert(tokens, Token.new("LessEqual"))
                --consome '='
                self:consume()
            else
                --adiciona na lista de tokens
                table.insert(tokens, Token.new("Less"))
            end
        elseif self:check('(') then
            table.insert(tokens, Token.new("OpenParen"))
            self:consume()
        elseif self:check(')') then
            table.insert(tokens, Token.new("CloseParen"))
            self:consume()
        elseif self:check('{') then
            table.insert(tokens, Token.new("OpenKey"))
            self:consume()
        elseif self:check('}') then
            table.insert(tokens, Token.new("CloseKey"))
            self:consume()
        elseif self:check('[') then
            table.insert(tokens, Token.new("OpenBracket"))
            self:consume()
        elseif self:check(']') then
            table.insert(tokens, Token.new("CloseBracket"))
            self:consume()
        elseif self:check('.') then
            table.insert(tokens, Token.new("Dot"))
            self:consume()
        elseif self:check(':') then
            table.insert(tokens, Token.new("Colon"))
            self:consume()
        elseif self:check(',') then
            table.insert(tokens, Token.new("Comma"))
            self:consume()
        else
            error("caracter não conhecido '" .. self:peek() .. "'")
        end
    end

    ---isso é para debug, para mostrar os tokens criados
    -- for _, token in ipairs(tokens) do
    --     if token.value then
    --         print("Token(kind = " .. token.kind .. ", value = " .. token.value .. ")")
    --     else
    --         print("Token(kind = " .. token.kind .. ")")
    --     end
    -- end

    return tokens
end
return Lexer